<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="theme-color" content="#FFDFD6" />
    <title>Directory Listing</title>
    <link rel="icon" type="image/x-icon" href="https://muzu.me/filesicon.png">
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/masonry/4.2.2/masonry.pkgd.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/imagesloaded/4.1.4/imagesloaded.pkgd.min.js"></script>
    <style>
        body {
            background-color: #ffffff;
            color: #1A202C;
            font-family: sans-serif;
        }
        header {
            background-color: #f7fafc;
            text-align: center;
            padding: 1rem;
        }
        header img {
            max-width: 100%;
            height: auto;
        }
        ul {
            list-style-type: none;
            padding: 0;
            margin: 0;
        }
        .file-item {
            margin-bottom: 1rem;
            position: relative;
        }
        .file-item a {
            display: flex;
            align-items: center;
            padding: 0.5rem 1rem;
            background-color: #f7fafc;
            border-radius: 0.5rem;
            text-decoration: none;
            color: #1A202C;
            transition: background-color 0.3s, box-shadow 0.3s;
            position: relative;
        }
        .file-item a:hover {
            background-color: #e2e8f0;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .file-item a[style] {
            color: white;
        }
        .icon {
            width: 2rem;
            height: 2rem;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 1rem;
            background-color: #e2e8f0;
            border-radius: 0.5rem;
        }
        .icon-directory::before { content: "\1F4C2"; }
        .icon-file::before { content: "\1F4C4"; }
        .icon-image::before { content: "\1F5BC"; }
        .icon-archive::before { content: "\1F4E6"; }
        .icon-text::before { content: "\1F4C3"; }
        .icon-video::before { content: "\1F4FA"; }
        .icon-rar::before { content: "\1F4E6"; }
        .icon-apk::before { content: "\1F4F1"; }
        .icon-torrent::before { content: "\1F4C3"; }
        .icon-exe::before { content: "\1F5A5"; }
        .icon-dmg::before { content: "\1F4BE"; }
        .icon-audio::before { content: "\1F3B5"; }
        .icon-pinned {
            position: absolute;
            right: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: red;
        }
        .icon-pinned::before {
            content: "\1F4CC";
        }
        .locked a {
            pointer-events: none;
            opacity: 0.5;
        }
        .locked .icon::before {
            content: "\1F512";
        }
        .details {
            flex: 1;
        }
        .name {
            font-weight: bold;
        }
        .type {
            color: #4A5568;
        }
        .size {
            color: #A0AEC0;
        }
        footer {
            background-color: #ffffff;
            color: #4A5568;
            text-align: center;
            padding: 1rem;
            font-size: 0.875rem;
            position: inline;
            bottom: 0;
            width: 100%;
            border-top: 1px solid #e2e8f0;
        }
        .note-container {
            background-color: #f7f7f7;
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 16px;
        }
        .note-title {
            font-size: 1.2em;
            font-weight: bold;
            margin-bottom: 8px;
        }
        .note-content {
            white-space: pre-wrap;
        }
        .gallery-container {
            background-color: #f7f7f7;
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 16px;
        }
        .gallery {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            grid-gap: 10px;
        }
        .gallery-item {
            background-color: #ffffff;
            border: 1px solid #e2e8f0;
            border-radius: 4px;
            overflow: hidden;
            transition: box-shadow 0.3s ease;
        }
        .gallery-item:hover {
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        .gallery-item img {
            width: 100%;
            height: 150px;
            object-fit: cover;
            display: block;
        }
        .gallery-item .text {
            padding: 8px;
            font-size: 14px;
            text-align: center;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        .gallery-folder {
            display: flex;
            align-items: center;
            justify-content: center;
            height: 150px;
            font-size: 48px;
            background-color: #f7fafc;
        }
        .audio-player {
            background-color: #f7fafc;
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 16px;
        }
        .audio-player audio {
            width: 100%;
        }
        .audio-list {
            list-style-type: none;
            padding: 0;
        }
        .audio-item {
            display: flex;
            align-items: center;
            padding: 8px;
            border-bottom: 1px solid #e2e8f0;
        }
        .audio-item:last-child {
            border-bottom: none;
        }
        .audio-item .icon {
            margin-right: 8px;
        }
        .audio-item .name {
            flex-grow: 1;
        }
        .audio-item .play-btn {
            background-color: #4299e1;
            color: white;
            border: none;
            border-radius: 4px;
            padding: 4px 8px;
            cursor: pointer;
        }
    </style>
    <script>
        function filterFiles() {
            const input = document.getElementById('searchBar');
            const filter = input.value.toLowerCase();
            const items = document.getElementById('fileList').getElementsByTagName('li');
            for (let i = 0; i < items.length; i++) {
                const item = items[i];
                const text = item.textContent || item.innerText;
                item.style.display = text.toLowerCase().indexOf(filter) > -1 ? '' : 'none';
            }
        }
        function lazyLoadImages() {
            const images = document.querySelectorAll('.gallery img');
            const options = {
                root: null,
                rootMargin: '0px',
                threshold: 0.1
            };
            const observer = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        img.src = img.dataset.src;
                        observer.unobserve(img);
                    }
                });
            }, options);
            images.forEach(img => observer.observe(img));
        }
        document.addEventListener('DOMContentLoaded', function() {
            var gallery = document.querySelector('.gallery');
            if (gallery) {
                imagesLoaded(gallery, function() {
                    new Masonry(gallery, {
                        itemSelector: '.gallery-item',
                        columnWidth: '.gallery-item',
                        percentPosition: true
                    });
                });
            }
            lazyLoadImages();
        });
        function playAudio(url) {
            const audioPlayer = document.getElementById('audioPlayer');
            audioPlayer.src = url;
            audioPlayer.play();
        }
    </script>
</head>
<body>
    <header>
        <div class="container mx-auto">
           <a href="/">
                <img src="https://muzu.me/muzu.svg" alt="Muzu Logo" class="w-3/4 mx-auto md:w-1/4">
            </a>
        </div>
    </header>
    <div class="container mx-auto p-4">
        <input type="text" id="searchBar" onkeyup="filterFiles()" placeholder="Search current directory..." class="w-full p-2 mb-4 bg-gray-100 text-gray-900 rounded-md">
    </div>
    <div class="container mx-auto p-4">
<?php
// Base directory
$baseDir = __DIR__;

// Get the current directory from the query string or default to the base directory
$currentDir = isset($_GET['dir']) ? realpath($baseDir . '/' . $_GET['dir']) : $baseDir;

// Ensure the directory is within the base directory
if (strpos($currentDir, $baseDir) !== 0 || !is_dir($currentDir)) {
    die('Invalid directory.');
}

// Function to get file type
function getFileType($file) {
    $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
    if (is_dir($file)) {
        return 'directory';
    } elseif (in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
        return 'image';
    } elseif (in_array($extension, ['txt', 'md'])) {
        return 'text';
    } elseif (in_array($extension, ['zip', 'tar', 'gz', 'rar'])) {
        return 'archive';
    } elseif (in_array($extension, ['mp4', 'avi', 'mov'])) {
        return 'video';
    } elseif (in_array($extension, ['mp3', 'wav', 'ogg', 'flac'])) {
        return 'audio';
    } elseif ($extension === 'apk') {
        return 'apk';
    } elseif ($extension === 'torrent') {
        return 'torrent';
    } elseif ($extension === 'exe') {
        return 'exe';
    } elseif ($extension === 'dmg') {
        return 'dmg';
    } else {
        return 'file';
    }
}

// Function to format file size
function formatSize($size) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $unit = 0;
    while ($size >= 1024 && $unit < count($units) - 1) {
        $size /= 1024;
        $unit++;
    }
    return round($size, 2) . ' ' . $units[$unit];
}

// Function to check for .private file
function isDirectoryLocked($dir) {
    return file_exists($dir . '/.private');
}

// Function to check for .pin file
function isDirectoryPinned($dir) {
    return file_exists($dir . '/.pin');
}

// Function to get accent color from a file in the directory
function getAccentColor($dir) {
    $files = scandir($dir);
    foreach ($files as $file) {
        if (preg_match('/^\.(#[0-9A-Fa-f]{6})$/', $file, $matches)) {
            return $matches[1]; // Return the hex color code
        }
    }
    return null;
}

// Function to check for .note file
function hasNoteFile($dir) {
    return file_exists($dir . '/.note');
}

// Function to read and format note content
function formatNoteContent($filePath) {
    $content = file_get_contents($filePath);
    $title = basename($filePath, '.txt');
    return "<div class='note-container'>
                <div class='note-title'>$title</div>
                <div class='note-content'>$content</div>
            </div>";
}

// Function to check for .gallery file
function hasGalleryFile($dir) {
    return file_exists($dir . '/.gallery');
}

// Function to check for .audio file
function hasAudioFile($dir) {
    return file_exists($dir . '/.audio');
}

// Generate base URL
$baseUrl = 'https://files.muzu.me';

// Check if we can open the directory
if (is_dir($currentDir) && $handle = opendir($currentDir)) {
    $entries = [];

    // Collect all entries
    while (false !== ($entry = readdir($handle))) {
        if ($entry != "." && $entry != ".." && $entry[0] != '.') { // Exclude dot files
            $filePath = $currentDir . '/' . $entry;
            $accentColor = is_dir($filePath) ? getAccentColor($filePath) : null;
            $entries[] = [
                'name' => $entry,
                'path' => $filePath,
                'relativePath' => str_replace($baseDir, '', $filePath),
                'type' => getFileType($filePath),
                'size' => is_dir($filePath) ? '-' : formatSize(filesize($filePath)),
                'locked' => is_dir($filePath) && isDirectoryLocked($filePath),
                'pinned' => is_dir($filePath) && isDirectoryPinned($filePath),
                'accentColor' => $accentColor
            ];
        }
    }
    closedir($handle);

    // Sort entries: alphabetically with pinned folders first, then locked entries at the end
    usort($entries, function($a, $b) {
        if ($a['pinned'] !== $b['pinned']) {
            return $a['pinned'] ? -1 : 1;
        }
        if ($a['locked'] !== $b['locked']) {
            return $a['locked'] ? 1 : -1;
        }
return strcasecmp($a['name'], $b['name']);
    });

    // Check if .gallery file exists
    $hasGalleryFile = hasGalleryFile($currentDir);

    // Check if .audio file exists
    $hasAudioFile = hasAudioFile($currentDir);

    echo "<ul id='fileList'>";
    // Add parent directory link if not in base directory
    if ($currentDir != $baseDir) {
        $relativeParentDir = str_replace($baseDir, '', dirname($currentDir));
        echo "<li class='file-item directory'>
                <a href='?dir=" . urlencode($relativeParentDir) . "'>
                    <div class='icon icon-directory'></div>
                    <div class='details'>
                        <div class='name'>..</div>
                        <div class='type'>Go up one level</div>
                    </div>
                </a>
            </li>";
    }

    // Display audio player and list if .audio file exists
    if ($hasAudioFile) {
        echo "<div class='audio-player'>
                <audio id='audioPlayer' controls>
                    Your browser does not support the audio element.
                </audio>
              </div>";
        echo "<ul class='audio-list'>";
        foreach ($entries as $entry) {
            if ($entry['type'] === 'audio') {
                echo "<li class='audio-item'>
                        <div class='icon icon-audio'></div>
                        <div class='name'>{$entry['name']}</div>
                        <button class='play-btn' onclick='playAudio(\"{$baseUrl}{$entry['relativePath']}\")'>Play</button>
                      </li>";
            }
        }
        echo "</ul>";
    }

    // Display gallery if .gallery file exists
    if ($hasGalleryFile) {
        echo "<div class='gallery-container'>";
        echo "<div class='gallery'>";
        foreach ($entries as $entry) {
            if ($entry['type'] === 'image') {
                echo "<div class='gallery-item'>
                        <a href='{$baseUrl}{$entry['relativePath']}'>
                            <img src='{$baseUrl}{$entry['relativePath']}' alt='{$entry['name']}' loading='lazy'>
                            <div class='text'>{$entry['name']}</div>
                        </a>
                      </div>";
            } elseif ($entry['type'] === 'directory') {
                echo "<div class='gallery-item'>
                        <a href='?dir=" . urlencode($entry['relativePath']) . "'>
                            <div class='gallery-folder'>📁</div>
                            <div class='text'>{$entry['name']}</div>
                        </a>
                      </div>";
            }
        }
        echo "</div>";
        echo "</div>";
    }

    // Display regular file list if not in gallery mode
    if (!$hasGalleryFile && !$hasAudioFile) {
        $hasNoteFile = hasNoteFile($currentDir);
        foreach ($entries as $entry) {
            $iconClass = 'icon-' . $entry['type'];
            $link = $entry['locked'] ? '#' : (is_dir($entry['path']) ? '?dir=' . urlencode($entry['relativePath']) : $baseUrl . $entry['relativePath']);
            $lockedClass = $entry['locked'] ? 'locked' : '';
            $pinnedClass = $entry['pinned'] ? 'icon-pinned' : '';
            $accentColorStyle = $entry['accentColor'] ? 'style="background-color: ' . $entry['accentColor'] . ';"' : '';

            if ($hasNoteFile && $entry['type'] === 'text' && pathinfo($entry['name'], PATHINFO_EXTENSION) === 'txt') {
                echo formatNoteContent($entry['path']);
            } else {
                echo "<li class='file-item $lockedClass directory'>
                        <a href='$link' $accentColorStyle>
                            <div class='icon $iconClass'></div>
                            <div class='details'>
                                <div class='name'>{$entry['name']}</div>
                                <div class='type'>{$entry['type']}</div>
                                <div class='$pinnedClass'></div>
                            </div>
                            <div class='size'>{$entry['size']}</div>
                        </a>
                    </li>";
            }
        }
    }

    echo "</ul>";
} else {
    echo "<p class='text-gray-500'>Unable to open directory.</p>";
}
?>
    </div>
    <footer>
        Muzu Directory v0.82_r1
    </footer>
</body>
</html>